/**
 *
 * \file        uart_cresnet_slave.c
 *
 * \brief       This is intended to be a generic module relating to UART usage
 *              for (serial/uart) Cresnet, slave end.
 *
 * \author      Pete McCormick
 *
 * \date        12/5/2007
 *
 * \note        Processor-specific stuff should be kept out of here.
 */

////////////////////////////////////////////////////////////////////////////////

#include "Cresnet.h"
#include "cresnet_slave.h"
#include "uart_cresnet.h"
#include "uart_cresnet_slave.hpp"
#include <string.h>                     // for memcpy
#include "os.h"                         // for OS_ISR_FLAG_NO_MORE_DATA
#include "dm_netmapper.hpp"
#include "hardware.h"
#include "errors.h"
#include "product.h"                    // for FirmwareGetName
#include "memorymanager.h"
#include "stdlib.h"
#include "cnet_pnp.h"
#include "StreamMgr.h"
#include "field_debug.h"
#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT   
#include <stdio.h>
#include "DMController.h"
#endif
//#include "dm_serialport.h"

////////////////////////////////////////////////////////////////////////////////
//#define CRESNET_PORTS_CNT               2


////////////////////////////////////////////////////////////////////////////////

// required Cresnet functions:

// UartCresnet refers to Cresnet on a serial port, i.e. not CIP/ethernet
// Made these into variables since they may change at runtime

// We are set up to buffer multiple cresnet packets, could be done in a more
// memory-efficient way
// Each Stream has its own Cresnet ID and therefore has to have its own data
//

////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////


// pre-built cresnet packets
static const UINT8 CresnetRestartMsg[] = {'\x2','\x2','\x3','\x0'};
static const UINT8 CresnetToken[] = {CNET_ID_CTRLSYS,TOKEN_PASS};
//static const UINT8 CresnetEnablePacket[] = {BROADCAST,4,PORT_FORCE,'C','E','\r'};
// DM link broken broadcast message
static const UINT8 BroadcastOffline[] = {BROADCAST,0x03,INTERNAL,INTERNAL_SUBCMD_DMLINK_BROKEN,0x00};

CDMUartCresnetSlave *pUartCresnetSlave[MAX_SLAVE_PORTS] = {0,0};
CDMUartCresnetSlave *pMidpointUart = 0;
void (*pfProjectMidpointOnlineStatus)(UINT8) = NULL;

void ByteCopy(volatile UINT8 *destP, volatile UINT8 *srcP, UINT16 count)
{
    UINT16 index;
    for (index = 0; index < count; index++)
        *destP++ = *srcP++;
}


////////////////////////////////////////////////////////////////////////////////
//  Callback functions

/**
 * \author      Larry Salant
 * \date        1/25/2009
 * \brief       watchdog timer for activity on the Midpoint slave port to determine if
 *              attached enpoint (or repeater) is online
 * \param       none
 * \return      none
 * \retval      void
 */
void MidpointTimer(void)
{
    if (pMidpointUart)
      pMidpointUart->MidpointOnlineTimer();
}

void SlaveTimerIsr(UINT32 pUartSlave)
{
    ((CDMUartCresnetSlave *)pUartSlave)->UartCresnetSlaveTimerIsr();
}


/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveGetPPNAddr
 * \return      none
 * \retval      void
 * \param       void
 */
UINT32 UartCresnetSlaveGetPPNAddr(void)
{
    return GetTSID();
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       save the tsid in NVL and in the uart cresnet slave
 * \return      none
 * \retval      void
 * \param       void
 */
void SetPNPNumber(UINT32 tsid)
{
  // from stream 0, send identity
  if (GetCresnetSlaveUart())
    GetCresnetSlaveUart()->UartCresnetSlaveSetIdentity(0);

  SaveTSID(tsid);
}

/**
 * \author      Larry Salant
 * \date        1/20/2009
 * \brief       Returns the pointer to the Cresnet (or DM Net Slave)class
 * \param       void
 * \return      CDMUartCresnetSlave *
 * \retval      void
 */
CDMUartCresnetSlave *GetCresnetSlaveUart(void)
{
  return pUartCresnetSlave[DMNET_SLAVE];
}
/**
 * \author      Larry Salant
 * \date        1/20/2009
 * \brief       Returns the pointer to the DM Net Midpoint slave uart
 *              currently, this is only used on the DM Input of a DM switch
 * \param       void
 * \return      CDMUartCresnetSlave *
 * \retval      void
 */
CDMUartCresnetSlave *GetDMMidpointUart(UINT8 streamId)
{
  // For the card firmware the streamId passed in is ignored. Always return back
  // the entry at index DMNET_MIDPOINT.
  return pUartCresnetSlave[DMNET_MIDPOINT];
}

////////////////////////////////////////////////////////////////////////////////

UINT8 CDMUartCresnetSlave::UartCresnetGetSlaveId( void )
{
    //Return the UART Cresnet slave ID
    return( m_id );
}

void CDMUartCresnetSlave::UartCresnetSlaveTx(UINT8 data)
{
    HwUartTx(m_uart, data);
}

/**
 * \author      Larry Salant
 * \date        5/6/2008
 * \brief       check if the queue for this stream is filled
 * \param       void
 * \return      UINT8
 * \retval      0 if the queue has room
 */
UINT8 CDMUartCresnetSlave::IsUartCrestnetSlaveQueueFull(UINT8 bStream)
{
    SLAVESTREAM *pStream = m_pStream;
    // set pointer to the queue for this stream
    if (bStream < m_bNumStreams)
      pStream += bStream;
    // if queue is full
    return pStream->pOutGoingCmdQueue->IsFull();
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \return      none
 * \retval      void
 * \brief
 * \param       void
 */
UINT8 CDMUartCresnetSlave::UartCresnetSlaveTxPacketReady(UINT8 bStream, UINT8 *pMessage)
{
    SLAVESTREAM *pStream = m_pStream;
    // set pointer to the queue for this stream
    if (bStream < m_bNumStreams)
        pStream += bStream;
    // if queue is full
    if (pStream->pOutGoingCmdQueue->IsFull())
    {
        //Bugzilla 30086 fix
        //Set error code to skip prints during CresnetConsole prints b/c infinite loop will occur
        if ( pMessage[2] == RCON_PKT )
            DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_OUT_QUEUE_FULL,DM_ERROR_CAUSE_CODE_DO_NOT_PRINT);
        else
            DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_OUT_QUEUE_FULL,bStream);

        // free buffer
        MemMgr->FreeBlock(pMessage);

        //Return unsuccessful status
        return false;
    }
    else // add this message to the queue
    {
#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT   
        if (IsDmFieldDebugIndexActive(DM_FIELD_CRESNET_TX_IDX))
        {
            char txName[5];
            sprintf(txName, "Tx%d:",bStream);
            print_packet((char *)pMessage, txName);  
        
        }
#endif
        pStream->pOutGoingCmdQueue->WriteLong((UINT32)pMessage);

        //Return successful status
        return true;
    }
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveSetID
 * \return      none
 * \retval      void
 * \param       id
 */
void CDMUartCresnetSlave::UartCresnetSlaveSetID(UINT8 id)
{
    if(id != m_id)
    {
        m_id = id;
        CresnetParamsSave(id);
    }
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveInit
 * \return      none
 * \retval      void
 * \param       bNumStreams - number of streams on the card (each stream is a consecutive address)
 */
#ifndef REMOVE_DM_ENCAP_HELPER
CDMUartCresnetSlave::CDMUartCresnetSlave(UINT8 bNumStreams, UINT8 port, UINT8 uart, UINT8 timer, UINT8 cresnetId, UINT8 bRxEnableNeedCardReady,
                                         UINT8 outQueueLength, UINT8 inMidpointQueueLength, BOOL SendAutoID,
                                         DmEncapChecksumHelper* pHelper /* = 0 */, CDMNetMapper *tmpDMNetMapper /* = pDMNetMapper*/):
#else
CDMUartCresnetSlave::CDMUartCresnetSlave(UINT8 bNumStreams, UINT8 port, UINT8 uart, UINT8 timer, UINT8 cresnetId,UINT8 bRxEnableNeedCardReady,
                                         UINT8 outQueueLength, UINT8 inMidpointQueueLength, BOOL SendAutoID,
                                         void* pHelper /* = 0 */, CDMNetMapper *tmpDMNetMapper /* = pDMNetMapper*/):
#endif
    m_MidData(NULL),
    m_bitsPerSec(38400),
    m_charTimeUsec((BITS_PER_CHAR * 1000000)/38400),
    m_rxDataIdx(0),
    m_rxWriteBuf(0),
    m_netmode(0),
    m_netmodeNext(0),
    m_txBytes(0),
    m_txPos(0),
    m_sendPtr(NULL),
    m_restart(0),
    m_pktInProgress(0),
    OfflineTimeout(0),
    m_bRespondToPoll(TRUE),
    m_bTxEnable(TRUE),
#ifdef CNET_STATS
    m_isrCnt(0),
    m_timerCnt(0),
    m_breakCnt(0),
    m_rxByteCnt(0),
    m_txByteCnt(0),
    m_txReadyCnt(0),
    m_pollCnt(0),
    m_overrunCnt(0),
#endif
    m_sync(FALSE),
    m_bypassMapper(FALSE),
    m_pfUartCresnetFwdMsg(NULL),
    m_pDMNetMapper(tmpDMNetMapper)
{
    SLAVESTREAM *pStream;

    if(!m_pDMNetMapper)
    {
        DmSystemError(DM_ERROR_LEVEL_FATAL,DM_ERROR_SUBSYS_CNET,ERR_CNET_DMNET_MAPPER_UNINITIALIZED, 0);
    }

    DmConsolePrintf("UartCresnetSlave(%d): timer %d, uart %d\r",m_pDMNetMapper->m_iDmNet, timer,uart);

#ifndef REMOVE_DM_ENCAP_HELPER
    //Save the helper
    m_pEncapChecksumHelper = pHelper;
#endif

//    //Assume we can respond to polls
//    m_bRespondToPoll = true;
//
//    //Default is to use the mapper
//    //For DGE-2 the global mapper is used by the DM net Master
//    m_bypassMapper = false;
//    m_pfUartCresnetFwdMsg = NULL;

    // save the pointer to the class
    pUartCresnetSlave[port] = this;

    // save uart and timer
    m_uart = uart;
    m_timer = timer;
    m_port = port;

    // set id of this device
    m_id = cresnetId;
    m_bNumStreams = bNumStreams;

    // disable Tx
    UartCresnetSlaveDropBus(m_uart);
//    m_restart = 0;
//    m_netmode = 0;
//    m_netmodeNext = 0;

    // set id of this device
    UartCresnetSlaveSetID(cresnetId);

    //UartCresnetSlave.maxRedirectMsgSize = 240;
    // initialize buffer pointers
//    m_rxWriteBuf = 0;
//    m_txBytes = 0;
//    m_txPos = 0;
//    m_rxDataIdx = CRESNET_BUF_ID_OFFSET;      // skip 1st byte - leave for source

    // allocate memory for each stream specific data
    pStream = (SLAVESTREAM *)malloc(sizeof(SLAVESTREAM)*bNumStreams);
    DmFatalOnZeroPointer((void*)pStream,DM_ERROR_SUBSYS_CNET,ERR_CNET_NO_MEMORY,0);

    m_pStream = pStream;
    // initialize the stream specific data
    for ( int stream=0; stream<bNumStreams; stream++ )
    {
        if ( SendAutoID )
        {
            // set the auto id flag so we send our id when we start, used for DM cards identifications toward the switch
            pStream->autoid = 1;
        }
        else
        {
            pStream->autoid = 0;
        }
        pStream->sendIdentity = 0;

        pStream->pOutGoingCmdQueue = new CBuffer(outQueueLength, CBUFFER_WIDTH_POINTER);
        // update pointer to the next stream
        pStream++;
    }

    //if this is device can act as a midpoint
    // (e.g input or output card)
    if (m_pDMNetMapper->CanBeMidpoint(port))
    {
      m_MidData = (MIDPOINT_DATA *)malloc(sizeof(MIDPOINT_DATA));
      if (!m_MidData)
	  {
        DmSystemError(DM_ERROR_LEVEL_FATAL,DM_ERROR_SUBSYS_CNET,ERR_CNET_NO_MEMORY,1);
		return;
	  }

      //initialize midpoint port offline
      m_MidData->m_EndpointOnline = 0;
      m_MidData->m_pPendingToController = new CBuffer(inMidpointQueueLength, CBUFFER_WIDTH_POINTER);;

      // start a periodic timer to monitor the status of the endpoint
      // (checks if we are being polled periodically by the endpoint)
      pMidpointUart = this;
      OsStartTimer(MIDPOINT_TIMER_PERIOD, (void*)MidpointTimer, OS_TIMER_PERIODIC);
    }
    else
      m_MidData = 0;

	if(bRxEnableNeedCardReady)
	{
		Reconfigure(false);
	}
	else
	{
    	Reconfigure(true);
	}

}

CDMUartCresnetSlave::~CDMUartCresnetSlave()
{
    if (m_pStream)
    {
        free(m_pStream);
        m_pStream = NULL;
    }
	if (m_MidData)
	{
		free(m_MidData);
		m_MidData = NULL;
    }

}
// processing called from Cresnet rx hisr
INT32 CDMUartCresnetSlave::UartCresnetSlavePPNImmediateProcessing(UINT8 * packet, UINT8 id, BOOL * pRecursed, BOOL * pReturn)
{
  BOOL retval;
  UINT8 in_packet_length;

  *pReturn = 0;
  // if PNP not supported, quit
  if (!pfCheckTouchSettableIDPacket)
     return 0;

  if (!*pRecursed && id == CNET_ID_PPN_ADDRESSING)
  {
    if ((*pfCheckTouchSettableIDPacket)(packet) == 2)
    {
      // modify the packet in place
      in_packet_length = packet[1];
      memmove(&packet[2],&packet[8],in_packet_length - 6);
      packet[0] = UartCresnetSlaveGetID();
      packet[1] = in_packet_length - 6;
      *pRecursed = 1;
      retval = UartCresnetSlaveValidateLocalMsg(packet);  // call here agin recursively
      *pRecursed = 0;
      *pReturn = 1;
      return retval;
    }
    else
    {
      *pReturn = 1;
      return 0;
    }
  }

  if (packet[2] == PNP_RESPOND_NOW)
  {
    if (packet[1] == 2)
    {
      if (packet[3] == 1)
      {
        UartCresnetSlaveGenerateBreak();
      }
      else if (packet[3] == 2)
      {
        UartCresnetSlaveSetIdentity(0);
        UartCresnetSlaveDisableReceiver();                              // disable receiver
                UartCresnetSlaveGrabBus(m_uart);              // turn off rcvr, turn on txmtr
        UartCresnetSlaveSendPacket(UartCresnetSlaveGetID());          // and send out our response

      }
    }
    *pReturn = 1;
    return 0;
  }

  return 0;
}

UINT8 CDMUartCresnetSlave::UartCresnetSlaveGetID(void)
{
  return m_id;
}

/**
 * \author      Larry Salant
 * \date        4/25/2008
 * \brief       check if this is our cresent id
 * \return      bool
 * \retval      1 = its our id
 * \param       UINT8 id
 */
UINT8 CDMUartCresnetSlave::UartCresnetSlaveIsOurID(UINT8 id)
{
    UINT8 match = 0;

    if (id >= m_id &&
        id <= (m_id + m_bNumStreams-1))
        match = 1;
    return match;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveBuildUpdateRequest
 * \return      none
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveBuildUpdateRequest(void)
{
    /* if we just powered up, tell ctl system */
    m_txBytes = sizeof(CresnetRestartMsg);
    m_sendPtr = (UINT8*)CresnetRestartMsg;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveBuildIdMsg
 * \return      none
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveBuildIdMsg(UINT8 bStream)
{
        UINT8 length;
    UINT8 * idmsg = m_immediatePkt;

    if(!idmsg)
    {
        return;
    }

    /* The 5th character is the start of the actual string */
#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT    
    FirmwareGetName((char *)&idmsg[5], CNET_MAX_MSG_SIZE-6, m_pDMNetMapper->m_iDmNet);
#else
    FirmwareGetName((char *)&idmsg[5], CNET_MAX_MSG_SIZE-6, bStream);
#endif

    length = strlen((char *)&idmsg[5]) +  3;

    /* now fill in the beginning of the packet */
    idmsg[0]= CNET_ID_CTRLSYS;
    idmsg[1]= length;
    idmsg[2]= NAME_REQ;
    idmsg[3]= 0;
    idmsg[4]= 0;
    m_sendPtr = idmsg;
    m_txBytes=(UINT16)(idmsg[CNET_LEN_OFFSET]+2);  /* calc # of chars to xmit */
}

/**
* \author      Pete McCormick
* \date        3/12/2008
* \brief       UartCresnetSlaveBuildIdMsg
* \return      none
* \retval      void
* \param       void
*/
void CDMUartCresnetSlave::UartCresnetSlaveBuildIdentityResponse(UINT8 bStream)
{
  UINT8 * IDENTITY_response = m_immediatePkt;
  INT32 length;
  char * pDest;
  UINT32 PPNAddress = UartCresnetSlaveGetPPNAddr();

  if(IDENTITY_response)
  {

    pDest = (char*)&IDENTITY_response[13];
    FirmwareGetName(pDest, CNET_MAX_MSG_SIZE-14,bStream);
    length = strlen(pDest) + 3;
    /* build the IDENTITY response */
    IDENTITY_response[0]  = CNET_ID_CTRLSYS;
    IDENTITY_response[1]  = (UINT8)(length + 8);
    IDENTITY_response[2]  = PNP_IDENTIFY_RESPONSE;
    // need to send as Big Endian but we can't use the swap macro because
    // we need to place in buffer on a non-word aligned boundary so copy as bytes.
    pDest = (char *)&PPNAddress;
    IDENTITY_response[6] = *pDest++;
    IDENTITY_response[5] = *pDest++;
    IDENTITY_response[4] = *pDest++;
    IDENTITY_response[3] = *pDest++;
    IDENTITY_response[7]  = 0x00;
    IDENTITY_response[8]  = PNP_IDENTIFY_RESPONSE;
    IDENTITY_response[9]  = m_id + bStream;
    IDENTITY_response[10] =
      (m_id == CNET_ID_PPN_ADDRESSING) ? TOKEN_PASS : BROADCAST;
    IDENTITY_response[11] = 0;
    IDENTITY_response[12] = 0;

    m_sendPtr = IDENTITY_response;
    m_txBytes = IDENTITY_response[CNET_LEN_OFFSET]+2;
  }
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveGetNextTxPacket
 * \return      INT32
 * \retval      -1 = nothing to send; otherwise, pointer to buffer
 * \param       bStream - which stream is sending data
 */
INT32 CDMUartCresnetSlave::UartCresnetSlaveGetNextTxPacket(UINT8 bStream)
{
        INT32 Status = -1;  // assume nothing to send
    SLAVESTREAM *pStream = m_pStream;
    // set pointer to the queue for this stream
    pStream += bStream;
    // if queue is not Empty
    if (!pStream->pOutGoingCmdQueue->IsEmpty())
    {
      // get next message from the queue and make it the active
      // outgoing message (note that only one message can go out at a
      // time from any stream so that's why there's only 1 count and send pointer
      m_sendPtr = (UINT8 *)pStream->pOutGoingCmdQueue->ReadLong();
      if (m_sendPtr)
      {
        m_txBytes = m_sendPtr[CNET_LEN_OFFSET] + 2;

        // indicate we have a message to send
        Status = 0;
      }
    }
    return Status;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveWaitAfterBreak
 * \return      none
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveWaitAfterBreak(void)
{
    while(!UartCresnetSlaveFifoIsEmpty(m_uart) || UartCresnetSlaveUartIsBusy(m_uart))
    {
    }
}

void CDMUartCresnetSlave::UartCresnetSlaveResetTimer(void)
{
    HwDelayUsecAbort(m_timer);
}

void CDMUartCresnetSlave::UartCresnetBreak(UINT32 inst, BOOL begin)
{
    // initiate a BREAK
    if(begin)
        HwUartSendBreak(inst);
    else
        HwUartClearBreak(inst);
}


void CDMUartCresnetSlave::UartCresnetSlaveSetNetmode(UINT8 netmode)
{
    if(m_netmode != netmode)
    {
        m_netmode = netmode;
    }
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       do something depending on our cresnet "state"
 * \return      none
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveTimerIsr(void)
{

    UartCresnetSlaveResetTimer();

#ifdef CNET_STATS
    m_timerCnt++;
#endif
    switch(m_netmode)
    {
        case CNET_SENDTOKEN:
            m_txBytes = sizeof(CresnetToken); // token is always 2 characters
            m_sendPtr = (UINT8*)CresnetToken;
        case CNET_START:                      // both modes function the same here
                        m_txPos = 0;                     // init index for char to send
#ifdef CNET_STATS
            m_txByteCnt++;  // for stats
#endif
                        UartCresnetSlaveTx(m_sendPtr[m_txPos++]);  // call 1st char routine
            m_txBytes--;                   // decrement # of chars left to send
                        break;

            // The STR9 CPU does not use the transmit UART FIFO. The transmit buffer is only two chars deep.
                        // The holding register and the shift register.
        case CNET_WAIT_AFTER_SEND:
                        //      wait for fifo to be empty and shift register to send the full byte, then proceed with the next state
                        if (UartCresnetSlaveFifoIsEmpty(m_uart) && !UartCresnetSlaveUartIsBusy(m_uart))
            {
                // FIFO and shift register are empty, go to next state in one char time
                UartCresnetSlaveSetNetmode(m_netmodeNext);
                UartCresnetSlaveTimeNet(m_charTimeUsec);
            }
            else
            {
                // otherwise wait about 1 bit time and check again
                UartCresnetSlaveTimeNet(m_charTimeUsec/10);
            }
            break;

        case CNET_RELEASE_BUS:
            UartCresnetSlaveDropBus(m_uart);                                  // disable Tx
                        UartCresnetSlaveEnableReceiver();                                                                 // enable receiver
            break;

        case CNET_WAIT_BEFORE_BREAK:
            UartCresnetSlaveDisableReceiver();                                                                  // Disable receiver
            UartCresnetBreak(m_uart, 1);
            UartCresnetSlaveGrabBus(m_uart);                                 // enable Tx
            UartCresnetSlaveSetNetmode(CNET_WAIT_DURING_BREAK);
            UartCresnetSlaveTimeNet(m_charTimeUsec+m_charTimeUsec/10);       // break durataion = 1+ character time
            break;

        case CNET_WAIT_DURING_BREAK:
            UartCresnetSlaveNetLed(0);
            UartCresnetSlaveDropBus(m_uart);                                  // disable Tx
            UartCresnetBreak(m_uart, 0);
            UartCresnetSlaveTimeNet(m_charTimeUsec);
            UartCresnetSlaveSetNetmode(CNET_WAIT_AFTER_BREAK);                // wait 1 character time before allowing input
            break;

        case CNET_WAIT_AFTER_BREAK:
            UartCresnetSlaveWaitAfterBreak();
                        UartCresnetSlaveEnableReceiver();                                                               // Enable receiver
            break;

        default:
            break;
    }
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       cause a timer to expire in the specified number of microseconds aka TimeNet
 * \return      none
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveTimeNet(UINT32 usec)
{
    HwDelayUsec(m_timer, usec, SlaveTimerIsr, (UINT32)this);
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveSendPacket
 * \return      none
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveSendPacket(UINT8 id)
{
    UINT32 status;
    UINT8 stream = GetStreamFromPacket(id);
    SLAVESTREAM *pStream = m_pStream;
    if (stream < m_bNumStreams)
      pStream += stream;

    if(pStream->sendIdentity)
    {
        UartCresnetSlaveBuildIdentityResponse(stream);
        pStream->sendIdentity = 0;
    }
    else if(pStream->autoid)
    {   /* auto id overrides all else */
        UartCresnetSlaveBuildIdMsg(stream);
        // if multiple streams, setup to send the next one
        pStream->autoid = 0;
    }
    else if(m_restart)
    {
        UartCresnetSlaveBuildUpdateRequest();
        m_restart = 0;
    }
    else
    {
        // check if there is a regular cresnet data packet ready
        status = UartCresnetSlaveGetNextTxPacket(stream);
        if(status == 0)
        {
            // got something to send
            m_pktInProgress = 1;
        }
        else
        {
            /* we have nothing to send, so just pass the token */

            // NOTE: The Cresnet spec calls for waiting one character time after sending a packet before
            // sending either the token or another packet. However, we need to remember that the UART has
            // a dual-buffered FIFO, which means that when we write a character to be output, it doesn't
            // actually go out the door for a full character time.  Therefore, we accomodate for that by
            // waiting for two character times here.

            UartCresnetSlaveSetNetmode(CNET_SENDTOKEN);
            UartCresnetSlaveTimeNet(m_charTimeUsec*2);
            return;
        }
    }

//PACKET_SENT:
    UartCresnetSlaveSetNetmode(CNET_START);  /* tell timer isr what to do */
    UartCresnetSlaveTimeNet(m_charTimeUsec*2); // wait 2 char times (effectively 1 char time - see above)
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief
 * \return      int
 * \retval      0 if packet has been consumed, or is not for us
 *              returns 1 if packet needs further processing packet points to id byte
 * \param       void
 */
int CDMUartCresnetSlave::UartCresnetSlaveValidateLocalMsg(UINT8 *packet)
{
    static BOOL recursed = 0;
    UINT8 id = *packet;
    UINT8 retval;
    BOOL returnNow;
    INT32 result;


    if(!UartCresnetSlaveIsOurID(id) && (id != BROADCAST)
        && (id != CNET_ID_PPN_ADDRESSING))
    {
        return 0;     /* packet is not for us */
    }

    /*-----------------------------------------------------------------
     * Check to see if it is a token pass.
     *-----------------------------------------------------------------*/
    if( *(packet+1) == TOKEN_PASS )
    {
        if(UartCresnetSlaveIsOurID(id))
        {
            ManageDMNET_LinkLed();
#ifdef CNET_STATS
            m_pollCnt++;
#endif
                        UartCresnetSlaveDisableReceiver();                      // disable receiver
            UartCresnetSlaveGrabBus(m_uart);           /* turn off rcvr, turn on txmtr */
            UartCresnetSlaveSendPacket(id);             /* and send out our response */
        }
        return 0;                    /* and we're done with this packet*/
    }

    /*-----------------------------------------------------------------
     * Check to see if it is an device ID request
     *-----------------------------------------------------------------*/
    if ( (id == BROADCAST) || UartCresnetSlaveIsOurID(id) )
    {
        if (    (packet[CNET_LEN_OFFSET] == 0x02)     // packet length
                && (packet[CNET_TYPE_OFFSET] == CNET_COMMAND)  // packet type
                && (packet[3] == CNET_CMD_NAME_REQ) )   // device ID request
        {
            // if its a broadcast,
            if  (id == BROADCAST)
            {
                retval = 1;
              // tell all streams to send their id
              for (id = 0; id < m_bNumStreams; id++)
                UartCresnetSlaveSetAutoId(id);
            }
            else // set auto id for the stream that was addressed
            {
              UartCresnetSlaveSetAutoId(GetStreamFromPacket(id));
              retval = 0;
            }
#ifdef CNET_STATS
            m_pollCnt++;
#endif
            return retval;                    /* and we're done with this packet*/
        }
    }

    result = UartCresnetSlavePPNImmediateProcessing(packet, id,  &recursed, &returnNow);
    if(returnNow)
    {
        return result;
    }

    // else it is okay
    return 1;
}
/**
* \author      Larry Salant
* \date        1/22/2009
* \brief       Midpoint timer to determine if Endpoint went offline
* \param       void
* \return      void
* \retval      none
*/
void CDMUartCresnetSlave::MidpointOnlineTimer(void)
{
    if ( m_bRespondToPoll )
    {
        // if the timer hasn't expired
        if (OfflineTimeout > 0)
        {
          // decrement timer and if timed out
          if (--OfflineTimeout == 0)
          {
            // set it offline
            MidpointSetOnLineStatus(FALSE);
          }
        }
    }
    else
    {
        //Initialize the timeout to timed out condition
        OfflineTimeout = 0;

        //Force midpoint offline immediately if needed
        if (m_MidData->m_EndpointOnline)
        {
            MidpointSetOnLineStatus(FALSE);
        }
    }
}
/**
* \author      Larry Salant
* \date        1/22/2009
* \brief       get pointer to packet to send to Slave Tx
* \return      UINT32 *
* \retval
* \param       void
*/
CREST_ALL_PACKETS * CDMUartCresnetSlave::MidpointGetNextPacket(void)
{
  CREST_ALL_PACKETS *pPtr = 0 ;

  // if there is a message pending
  if (m_MidData->m_pPendingToController && !m_MidData->m_pPendingToController->IsEmpty())
  {
      pPtr = (CREST_ALL_PACKETS *)m_MidData->m_pPendingToController->ReadLong();
  }

  return pPtr;
}

/**
* \author      Larry Salant
* \date        1/22/2009
* \brief       get pointer to packet to send to Slave Tx
* \return      UINT32 *
* \retval
* \param       void
*/
void CDMUartCresnetSlave::MidpointReleaseNextPacket(UINT8 *pBuffer)
{
  // free the buffer
  if (MemMgr && pBuffer)
  {
     MemMgr->FreeBlock(pBuffer);
  }
}
/**
* \author      Larry Salant
* \date        1/22/2009
* \brief       sets the state of the endpoint connected to the midpoint
*              (online/offline)
* \param       new state
* \return      void
* \retval
*/
void CDMUartCresnetSlave::MidpointSetOnLineStatus(UINT8 State)
{
    m_MidData->m_EndpointOnline = State;

#ifndef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT 
    // notify controller
    SendOnlineStatusToController(STREAM_1,m_MidData->m_EndpointOnline);
#else
    // send the online status command
     m_pDMNetMapper->MasterStatusChange(m_pDMNetMapper->m_iDmNet,State);
#endif

    // if offline,
    if (!State)
    {
        // also send a broadcast packet to the master.  This is needed if a repeater was
        // connected between a switch and the output card goes offline;  The repeater needs
        // to tell the input card.
        m_pDMNetMapper->SendBroadcastToEndpoint((UINT8*)&BroadcastOffline);
    }

#ifndef REMOVE_DM_ENCAP_HELPER
    //Bugzilla 44329 fix
    if ( m_pEncapChecksumHelper )
    {
        //Notify the encapsulation checksum helper the UART status
        m_pEncapChecksumHelper->SendUartOnlineStatus(State ? true : false);
    }
#endif

    if (pfProjectMidpointOnlineStatus)
        (*pfProjectMidpointOnlineStatus)(State);
}

/**
* \author      Larry Salant
* \date        1/22/2009
* \brief       if endpoint online, sent packet from controller to it
*              packet is either a broadcast or was encapsulated (one level of
*              encapsulation already removed by the parser)
* \param       pointer to packet to send
* \return      void
* \retval
*/
void CDMUartCresnetSlave::SendPacketToTxForEndpoint(UINT8 *pPacket)
{
  if (!m_bTxEnable || !m_MidData)
      return;

  // try to send the packet to the transmitter's queue
    UINT32 bTimeout = 2000;
  while (bTimeout > 0)
  {
    // if the endpoint is not online
    if (!m_MidData->m_EndpointOnline)
      break;   // quit and throw away the packet

    // if there's room in the output queue
    if (!IsUartCrestnetSlaveQueueFull(STREAM_1))
    {
      UINT16 lLength = (UINT16)((CREST_ALL_PACKETS *)pPacket)->generic.len + 2 ;

      // send the packet from the controller to it
      UartCresnetSlaveQueueTxPacket(STREAM_1, pPacket, lLength);
      // done
      break;
    }
    else
    {
       // give it a chance to empty the queue
       HwDelayMsec(5);
       bTimeout--;
    }
  }

    if(!bTimeout)
    {
        if ( pPacket[2] == RCON_PKT )
                DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_SLAVE_QUEUE_FULL_WAIT_TIMEOUT,DM_ERROR_CAUSE_CODE_DO_NOT_PRINT);
            else
                DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_SLAVE_QUEUE_FULL_WAIT_TIMEOUT,0);
    }

    return;
 }

/**
 * \author      Larry Salant
 * \date        2/1/2009
 * \brief       returns state of Midpoint Slave (slave port of a repeater or DM port
 *              of an input card)
 * \param       void
 * \return      bool
 * \retval      true if online
 */
bool CDMUartCresnetSlave::IsMidpointOnline(void)
{
  // if it's an midpoint, return status
  if (m_MidData)
    return m_MidData->m_EndpointOnline;
  else // e.g. output card
    return 1;
}

/**
 * \author      Larry Salant
 * \date        1/22/2009
 * \brief       called when the midpoint receives a packet from Endpoint or
 *              repeater.  Process appropriate broadcast commands and forward
 *              For other packets, encapsulate and send queue them to send to the
 *              controller
 * \param       void
 * \return      UINT32
 * \retval      isr flags
 */
UINT32 CDMUartCresnetSlave::UartCresnetMidpointRxIsr(void)
{
    UINT8 * packetPtr, *pBuffer;
    UINT32 flags = 0;
    UINT16 byteCnt;
    UINT8 id;

    //Bugzilla 41102 partial fix
    //Check if we should respond to polls.  THis was causing 10-11 online/offline packets to the switch and flooding our UART.
    if ( !m_bRespondToPoll )
        return 0;

    // Set online if not already
    if (!m_MidData->m_EndpointOnline)
        MidpointSetOnLineStatus(TRUE);

    //Check to see if it is a token pass.
    packetPtr = &m_rxData[m_rxWriteBuf][0];

#ifndef REMOVE_DM_ENCAP_HELPER
    //Check for the encapsulation checksum helper class
    if ( m_pEncapChecksumHelper )
    {
        //Check for an encapsulated checksum packet
        if ( m_pEncapChecksumHelper->IsEncapsulatedChecksumPacket( (UINT8*)&packetPtr[CRESNET_BUF_ID_OFFSET], (m_rxDataIdx-1) ) )
        {
            UINT8 bStatus;
            bStatus = m_pEncapChecksumHelper->VerifyPacket( (UINT8*)&packetPtr[CRESNET_BUF_ID_OFFSET], (m_rxDataIdx-1) );
            switch ( bStatus )
            {
                case DM_ENCAPSULATED_PACKET_STATUS_NO_ERRORS:
                case DM_ENCAPSULATED_PACKET_STATUS_ERROR_MISMATCH_SEQUENCE:
                    //Strip the packet
                    if ( m_pEncapChecksumHelper->RemoveEncapsulatedChecksumPacket( (UINT8*)&packetPtr[CRESNET_BUF_ID_OFFSET], (m_rxDataIdx-1) ) )
                    {
                        //Decrement the data byte count b/c the heaper was removed
                        m_rxDataIdx -= DM_ENCAPSULATED_CHECKSUM_PACKET_ADDITIONAL_BYTES;
                    }
                    break;

                case DM_ENCAPSULATED_PACKET_STATUS_ERROR_WRONG_TYPE:
                case DM_ENCAPSULATED_PACKET_STATUS_ERROR_INVALID_PARAMETERS:
                case DM_ENCAPSULATED_PACKET_STATUS_ERROR_INVALID_COUNT:
                case DM_ENCAPSULATED_PACKET_STATUS_ERROR_BAD_CHECKSUM:
                    //Log an error b/c we will throw it away
                    DmSystemError( DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_CNET, ERR_CNET_ENCAP_RX_DROP_PKT, bStatus );

                    //Reset the receive buffer
                    m_rxDataIdx = CRESNET_BUF_ID_OFFSET;

                    return flags;
            }
        }

        //Bugzilla 52637 fix
        //Name request timer should check after encapsulation is removed
        //Check if we need to look for a name request
        if ( m_pEncapChecksumHelper->IsNameRequestTimerNeeded() )
        {
            if ( (packetPtr[CRESNET_BUF_ID_OFFSET+1] > 3) &&          //LEN
                 (packetPtr[CRESNET_BUF_ID_OFFSET+2] == NAME_REQ) )   //CMD
            {
                //Stop the name request timer
                m_pEncapChecksumHelper->StopNameRequestTimer();
            }

            if ( (packetPtr[CRESNET_BUF_ID_OFFSET] == BROADCAST ) &&          //Broadcast
                 (packetPtr[CRESNET_BUF_ID_OFFSET+2] == DM_DEV_CONTROL_FLOW_REJECT_CMD) )   //CMD
            {
                //Bugzilla 44091 fix
                //When switches are cascaded, the input card will never receive a name request response back
                //Utilize the flow rejection packet as confirmation that the name request should be ignored
                m_pEncapChecksumHelper->StopNameRequestTimer();
            }
        }

        //Look for a sequence request or sequence response packet
        if ( ((m_rxDataIdx-1) == (MAX_DM_SEQUENCE_REQUEST_PACKET_BYTES)) ||
             ((m_rxDataIdx-1) == (MAX_DM_SEQUENCE_RESPONSE_PACKET_BYTES)) ||
             ((m_rxDataIdx-1) == (MAX_DM_LONG_REACH_PACKET_BYTES)))
        {
            switch ( packetPtr[CRESNET_BUF_TYPE_OFFSET] )
            {
                case DM_SEQUENCE_REQUEST_PACKET_CMD:
                case DM_SEQUENCE_RESPONSE_PACKET_CMD:
                case DM_LONG_REACH_PACKET_CMD:
                    return( UartCresnetSlaveRxIsr() );
            }
        }
    }
#endif

    id = packetPtr[CRESNET_BUF_ID_OFFSET];
    if (UartCresnetSlaveIsOurID(id) && (*(packetPtr+CRESNET_BUF_LEN_OFFSET) == TOKEN_PASS))
    {
        UartCresnetSlaveDisableReceiver();          // disable receiver
        UartCresnetSlaveGrabBus(m_uart);           /* turn off rcvr, turn on txmtr */
        // send out waiting packet, if any
        UartCresnetSlaveSendPacket(id);
    }
    else // process as needed and forward to controller
    {
        // if its a device ID request
        if ( (id == BROADCAST) || UartCresnetSlaveIsOurID(id) )
        {
            // validate message and perform any immediate processing required
            if (    (packetPtr[CRESNET_BUF_LEN_OFFSET] == 0x02)     // packet length
                    && (packetPtr[CRESNET_BUF_TYPE_OFFSET] == CNET_COMMAND)  // packet type
                    && (packetPtr[4] == CNET_CMD_NAME_REQ) )   // device ID request
            {
                // its an id Request, set flag to send ID on Midpoint's Tx
                // when next token is received by the Midpoint
                // if its a broadcast,
                if (id == BROADCAST)
                {
                    // tell all streams to send their id
                    for (id = 0; id < m_bNumStreams; id++)
                        UartCresnetSlaveSetAutoId(id);
                }
                else // set auto id for the stream that was addressed
                {
                    UartCresnetSlaveSetAutoId(GetStreamFromPacket(id));
                }
            }
            // TBD - other Broadcast commands which need to be processed?
        }
        // Send packet to the queue for the controller
        // get a memory buffer big enough in case we have to encapsulate it
        // This buffer will be released after the command is sent
        byteCnt = packetPtr[CRESNET_BUF_LEN_OFFSET] + 2;
        if (MemMgr && !m_MidData->m_pPendingToController->IsFull() && (pBuffer = MemMgr->GetBlock(byteCnt+4)) != 0)
        {
            // if its not a broadcast command, encapsulate it
            if (packetPtr[CRESNET_BUF_ID_OFFSET] != BROADCAST)
            {
#ifdef DEBUG
                if (packetPtr[CRESNET_BUF_TYPE_OFFSET] == CAPABILITY_RESPONSE)
                    DmSystemError(DM_ERROR_LEVEL_ERROR,0xFF,1, packetPtr[CRESNET_BUF_TYPE_OFFSET]);
#endif // DEBUG
                // add a level of encapsulation
                pBuffer[0] = CNET_ID_CTRLSYS;
                pBuffer[1] = byteCnt + 1; // +1 for the Controller ID packet
                pBuffer[2] = CRESNET_ENCAPSULATED;
                pBuffer[3] = DM_NET_ID;         // update device address
                ByteCopy(&pBuffer[4], &packetPtr[CRESNET_BUF_LEN_OFFSET], byteCnt-1);
            }
            else // copy broadcast message to buffer;
                ByteCopy(pBuffer, &packetPtr[CRESNET_BUF_ID_OFFSET], byteCnt);

            m_MidData->m_pPendingToController->WriteLong((UINT32)pBuffer);
#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
			m_pDMNetMapper->SendPendingToController(m_pDMNetMapper->m_iDmNet);              

#endif
        }
        else
            // error buffer not available ...
            // for now just reset buffer and throw out message
            DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_OUT_QUEUE_FULL, packetPtr[CRESNET_BUF_TYPE_OFFSET]);

    }
    // packet is consumed, reset receive buffer
    m_rxDataIdx = CRESNET_BUF_ID_OFFSET;

    return flags;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       triggered when there is something to parse
 * \return      UINT32
 * \retval
 * \param       void
 */
UINT32 CDMUartCresnetSlave::UartCresnetSlaveRxIsr(void)
{
    UINT8 * packetPtr;
    UINT32 flags = 0;
    UINT16 byteCnt;
    UINT8 *pBuffer;

    //Check if we should respond to polls
    if ( !m_bRespondToPoll )
        return 0;

    // Set online if not already
    if (m_MidData && !m_MidData->m_EndpointOnline)
      MidpointSetOnLineStatus(TRUE);

    packetPtr = &m_rxData[m_rxWriteBuf][0];

#ifndef REMOVE_DM_ENCAP_HELPER
    //Check for the encapsulation checksum helper class
    if ( m_pEncapChecksumHelper )
    {
        //Check for an encapsulated checksum packet
        if ( m_pEncapChecksumHelper->IsEncapsulatedChecksumPacket( (UINT8*)&packetPtr[CRESNET_BUF_ID_OFFSET], (m_rxDataIdx-1) ) )
        {
            UINT8 bStatus;
            bStatus = m_pEncapChecksumHelper->VerifyPacket( (UINT8*)&packetPtr[CRESNET_BUF_ID_OFFSET], (m_rxDataIdx-1) );
            switch ( bStatus )
            {
                case DM_ENCAPSULATED_PACKET_STATUS_NO_ERRORS:
                case DM_ENCAPSULATED_PACKET_STATUS_ERROR_MISMATCH_SEQUENCE:
                    //Strip the packet
                    if ( m_pEncapChecksumHelper->RemoveEncapsulatedChecksumPacket( (UINT8*)&packetPtr[CRESNET_BUF_ID_OFFSET], (m_rxDataIdx-1) ) )
                    {
                        //Decrement the data byte count b/c the heaper was removed
                        m_rxDataIdx -= DM_ENCAPSULATED_CHECKSUM_PACKET_ADDITIONAL_BYTES;
                    }
                    break;

                case DM_ENCAPSULATED_PACKET_STATUS_ERROR_WRONG_TYPE:
                case DM_ENCAPSULATED_PACKET_STATUS_ERROR_INVALID_PARAMETERS:
                case DM_ENCAPSULATED_PACKET_STATUS_ERROR_INVALID_COUNT:
                case DM_ENCAPSULATED_PACKET_STATUS_ERROR_BAD_CHECKSUM:
                    //Log an error b/c we will throw it away
                    DmSystemError( DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_CNET, ERR_CNET_ENCAP_RX_DROP_PKT, bStatus );

                    //Reset the receive buffer
                    m_rxDataIdx = CRESNET_BUF_ID_OFFSET;

                    return flags;
            }
        }

        //Look for a sequence request or sequence response packet or long reach packet
        if ( ((m_rxDataIdx-1) == (MAX_DM_SEQUENCE_REQUEST_PACKET_BYTES)) ||
             ((m_rxDataIdx-1) == (MAX_DM_SEQUENCE_RESPONSE_PACKET_BYTES)) ||
             ((m_rxDataIdx-1) == (MAX_DM_LONG_REACH_PACKET_BYTES)) )
        {
            switch ( packetPtr[CRESNET_BUF_TYPE_OFFSET] )
            {
                case DM_SEQUENCE_REQUEST_PACKET_CMD:
                case DM_SEQUENCE_RESPONSE_PACKET_CMD:
                case DM_LONG_REACH_PACKET_CMD:
                    packetPtr[CRESNET_BUF_ID_OFFSET] = m_pEncapChecksumHelper->GetPacketSlaveId();
                    break;
            }
        }
    }
#endif

    // validate message and perform any immediate processing required
    if ( (UartCresnetSlaveValidateLocalMsg(&packetPtr[CRESNET_BUF_ID_OFFSET]) > 0) &&
         (m_pfUartCresnetFwdMsg == NULL ) )
    {
        // stick the packet somewhere for later processing
        //  3/23/07 - Modified by HS to report broadcast packets
        if(packetPtr[CRESNET_BUF_ID_OFFSET] == BROADCAST)
        {
            packetPtr[CRESNET_BUF_SRC_OFFSET] = NET_SOURCE_CRESNET_BROADCAST;
        }
        else
        {
            packetPtr[CRESNET_BUF_SRC_OFFSET] = NET_SOURCE_CRESNET;
        }

        // get a temporary buffer big enough for the message (+2 is for id,len bytes)
        byteCnt = packetPtr[CRESNET_BUF_LEN_OFFSET] + 2 + CRESNET_BUF_ID_OFFSET;
        // add one more character incase someone adds a null terminator.
        if (MemMgr && (pBuffer = MemMgr->GetBlock(byteCnt+1)) != 0)
        {
          // copy message to buffer;  This buffer will be released after the command is
          // processed by the cresnet task.
          ByteCopy(pBuffer, packetPtr, byteCnt);

          // funnel to common processing
          flags |= CresnetSlaveAcceptIncomingPacket(pBuffer);
          // bump to next buffer, wrap
          m_rxWriteBuf++;
          // reset pointer within buffer
          m_rxDataIdx = CRESNET_BUF_ID_OFFSET;
          if(m_rxWriteBuf >= CRESNET_RX_BUF_CNT)
          {
              m_rxWriteBuf = 0;
          }
        }
        else
        {
            DmSystemError( DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_CNET, ERR_CNET_DROPPING_PKT, byteCnt+1 );
        }
    }

    // External processing for DGE-2 to send the data to the PC
    if (m_pfUartCresnetFwdMsg)
    {
      (*m_pfUartCresnetFwdMsg)( &packetPtr[CRESNET_BUF_ID_OFFSET] );
    }

    // else if packet is consumed, no need to go to next buffer
    m_rxDataIdx = CRESNET_BUF_ID_OFFSET;

    return flags;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       end of packet is indicated by 2nd byte of packet == 0
 *              or satisfied expected length for current packet
 *              returns next data byte for transmission, or 0xffff if done
 * \return      UINT32
 * \retval
 * \param       void
 */
UINT32 CDMUartCresnetSlave::UartCresnetSlaveIsr(UINT8 recstatus, BOOL rxReady, UINT8 newByte, BOOL txEmpty)
{
    UINT8 * pDest;
    UINT32 isrFlags = 0;
    UINT32 byteCnt;

    pDest = &m_rxData[m_rxWriteBuf][m_rxDataIdx];

#ifdef CNET_STATS
    m_isrCnt++;
#endif

    if(recstatus)
    {
#ifdef CNET_STATS
                if (recstatus == GOT_BREAK)     {
                        m_breakCnt++;
                }
#endif
        // received a break - reset our packet pointer
        m_rxDataIdx = CRESNET_BUF_ID_OFFSET;
        pDest = &m_rxData[m_rxWriteBuf][0];

        m_sync = FALSE;

        //      got break flag added, to set sync dmswitch ctlsys sends break periodically
        if ((recstatus == GOT_SYNC) || (recstatus == GOT_BREAK)) {
            m_sync = TRUE;
        }

#if 0   // Sending update request was causing OOBTF to lose card state

        //if we got an error trigger an update request

        else if( recstatus == GOT_ERROR ) {
            m_restart = 1;
        }
#endif

        rxReady = 0;
    }

    //Reset the timer used to determine if master is still present
    if ( m_bRespondToPoll )
        OfflineTimeout = ENDPOINT_WATCHDOG_TIMEOUT;

    // check for received data - packets are delimited by their length bytes
        if(rxReady && m_sync)
    {
#ifdef CNET_STATS
        m_rxByteCnt++;
#endif
        // got some cresnet data
        if(m_rxDataIdx == CRESNET_BUF_ID_OFFSET)
        {
            if(newByte != 0)
            {
                // very 1st byte in packet
                *pDest = newByte;
                m_rxDataIdx++;
            }
        }
        else if(m_rxDataIdx == CRESNET_BUF_ID_OFFSET+1)
        {
            // 2nd byte in packet - always the packet length
            *pDest = newByte;
            m_rxDataIdx++;
            if(newByte == TOKEN_PASS)
            {
                //Default is to use the mapper
                //For DGE-2 the global mapper is used by the DM net Master
                if ( m_bypassMapper )
                    UartCresnetSlaveRxIsr();
                else
                    isrFlags |= m_pDMNetMapper->ProcessRxPacket(m_port, 0);
            }
        }
        else
        {
            // check for buffer overrun
            if(m_rxDataIdx >= sizeof(m_rxData[0]))
            {
#ifdef CNET_STATS
                // count overruns
                m_overrunCnt++;
#endif
                // PEM - previous code did nothing to index?
                // which I think caused us to be stuck forever
                // just dump the data
                m_rxDataIdx = CRESNET_BUF_ID_OFFSET;
            }
            // buffer the current byte
            *pDest = newByte;
            m_rxDataIdx++;
            byteCnt = m_rxData[m_rxWriteBuf][CRESNET_BUF_LEN_OFFSET];
            // +2 is for id,len bytes
            if(m_rxDataIdx >= byteCnt + 2 + CRESNET_BUF_ID_OFFSET)
            {
                // we have satisfied the byte count for the current pending packet -
                // packet is now complete and needs to be parsed
                //
                //Default is to use the mapper
                //For DGE-2 the global mapper is used by the DM net Master
                if ( m_bypassMapper )
                    UartCresnetSlaveRxIsr();
                else
                    isrFlags |= m_pDMNetMapper->ProcessRxPacket(m_port, 0);
            }
        }
    }

    if(txEmpty)
    {
#ifdef CNET_STATS
        m_txReadyCnt++;
#endif
        // write next char in packet to duart, decrement # to go and check if done
        if(m_netmode == CNET_WAIT_AFTER_SEND)
        {
            return OS_ISR_FLAG_NO_MORE_DATA;   // done
        }
        if(m_txBytes)
        {
#ifdef CNET_STATS
            m_txByteCnt++;
#endif
            isrFlags |= m_sendPtr[m_txPos++];
            m_txBytes--;
            return isrFlags;       // not done, more to send
        }
        // if a data packet was just transmitted,
        if(m_pktInProgress)
        {
            m_pktInProgress = 0;

            // free the buffer (if it was dynamic)
            MemMgr->FreeBlock(m_sendPtr);
            m_sendPtr = 0;
        }
        // no more characters to send, so see what we need to do
        if (m_netmode == CNET_SENDTOKEN)
        {
            m_netmodeNext = CNET_RELEASE_BUS;
                }
        else
        {
            m_netmodeNext = CNET_SENDTOKEN;
                }

        // PEM - think this was 5272-specific
        UartCresnetSlaveSetNetmode(CNET_WAIT_AFTER_SEND);  // but first wait for fifo to empty
                UartCresnetSlaveTimeNet(m_charTimeUsec);           // and check every char
        isrFlags |= OS_ISR_FLAG_NO_MORE_DATA;

        return isrFlags;                                                                   // done
    }

    isrFlags |= OS_ISR_FLAG_NO_MORE_DATA;
    return isrFlags;        // done
}

void CDMUartCresnetSlave::UartCresnetSlaveSetTempID(UINT8 id)
{
    if(id != m_id)
    {
        m_id = id;
        //RecordAnalogChange(SYSTEM_CRESNET_ID, id);
    }
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveResetTxQueue
 * \return      none
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveResetTxQueue(UINT8 bStream)
{
    // just dequeue everything,
    // but don't do anything with the items

    SLAVESTREAM *pStream = m_pStream;
    UINT8 *pMessage;

    pStream += bStream;
    while (!pStream->pOutGoingCmdQueue->IsEmpty())
    {
      // get a buffer from the queue
      pMessage = (UINT8 *)pStream->pOutGoingCmdQueue->ReadLong();
      // free buffer
      MemMgr->FreeBlock(pMessage);
    }
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveSendToTxQueue
 *              MUST copy here since caller does not keep a copy
 *              of their packet - maybe they should!
 *              called from outside this module
 * \return      none
 * \retval      void
 * \param       void
 */
INT32 CDMUartCresnetSlave::UartCresnetSlaveSendToTxQueue(UINT8 bStream, UINT8 * packet)
{
    UINT8 bOriginalBytes = packet[CNET_LEN_OFFSET] + 2;
    UINT16 sAllocateBytes = bOriginalBytes; //Assume allocate bytes will be the same
    UINT8 bStatus;
    UINT8* pDest;

#ifndef REMOVE_DM_ENCAP_HELPER
    UINT8 bNeedEncapChecksum;

    //Check if we need to look for a name request
    if ( m_pEncapChecksumHelper && m_pEncapChecksumHelper->IsNameRequestTimerNeeded() )
    {
        //Check if this is a name request packet
        if ( (packet[1] == 0x02) &&
             (packet[2] == CNET_COMMAND) &&
             (packet[3] == CNET_CMD_NAME_REQ) )
        {
            m_pEncapChecksumHelper->StartNameRequestTimer();
        }
    }

    bNeedEncapChecksum = false;

    //Check if encapsulated checksum is needed
    if ( m_pEncapChecksumHelper && m_pEncapChecksumHelper->IsEncapsulatedChecksumEnabled() )
    {
        bNeedEncapChecksum = true;

        //Increase the # of bytes so encapsulation will fit
        sAllocateBytes += DM_ENCAPSULATED_CHECKSUM_PACKET_ADDITIONAL_BYTES;
        sAllocateBytes = min(CNET_MAX_MSG_SIZE, sAllocateBytes);
    }
#endif

    // get memory buffer;  Buffer will be released when message is sent
    if (!MemMgr || (pDest = MemMgr->GetBlock(sAllocateBytes)) == 0)
    {
#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT 
      if ( packet[2] == RCON_PKT )
            DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_DROP_PKT_SENDING,DM_ERROR_CAUSE_CODE_DO_NOT_PRINT);
        else
            DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_DROP_PKT_SENDING,2);
#else      
        //Bugzilla 30086 fix
        //Set error code to skip prints during CresnetConsole prints b/c infinite loop will occur
        if ( packet[2] == RCON_PKT )
            DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_DROP_PKT_SENDING,DM_ERROR_CAUSE_CODE_DO_NOT_PRINT);
        else
            DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_DROP_PKT_SENDING,2);
#endif
        return -1;
    }

    //Copy the original packet to the destination buffer
    ByteCopy(pDest, packet, bOriginalBytes);

#ifndef REMOVE_DM_ENCAP_HELPER
    //Apply encapsulated checksum if needed
    if ( bNeedEncapChecksum )
    {
        m_pEncapChecksumHelper->CreateEncapsulatedChecksumPacket(pDest, sAllocateBytes);
    }
#endif

    //Bugzilla 44001 fix
    //Attempt to send the packet and check the status
    bStatus = UartCresnetSlaveTxPacketReady(bStream, pDest);

#ifndef REMOVE_DM_ENCAP_HELPER
    if ( !bStatus && bNeedEncapChecksum )
    {
        //Transmission failed so decrement the local TX sequence
        m_pEncapChecksumHelper->DecrementLocalTxSequence();
    }
#endif

    return 0;
}


/**
 * \author      Adolfo Velasco
 * \date        01/16/2013
 * \brief       Handles processing when we stop responding to polls
 * \return      none
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::StopRespondingToPolls()
{
    //Clear the poll response flag
    m_bRespondToPoll = false;

    //Bugzilla 65915 fix
    //When 8G cable is disconnected the DMNet polling should stop
    //During this time, FDEBUG prints can fill up the queue and never empty since we are not responding to polls

    //Reset the transmit queue when DMNet polling is stopped to prevent this from happening
    for ( UINT32 i = 0; i < m_bNumStreams; i++ )
    {
        UartCresnetSlaveResetTxQueue(i);
    }
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveQueueTxPacket
 *              queue packet going out to cresnet must copy here!
 * \return      none
 * \retval      void
 * \param       void
 */
INT32 CDMUartCresnetSlave::UartCresnetSlaveQueueTxPacket(UINT8 bStream, void *packet, UINT32 packet_size)
{
    UINT32 status;
    UINT8 *pkt= (UINT8 *)packet;

    //Bugzilla 65915 fix
    //When 8G cable is disconnected the DMNet polling should stop
    //During this time, FDEBUG prints can fill up the queue and never empty since we are not responding to polls
    //Do not attempt to queue messages when polling is disabled
    if ( !m_bRespondToPoll )
    {
        return 0;
    }

    // validate stream
    if (bStream >= m_bNumStreams)
    {
        //Bugzilla 30086 fix
        //Set error code to skip prints during CresnetConsole prints b/c infinite loop will occur
        if ( pkt[2] == RCON_PKT )
            DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_INVALID_STREAM,DM_ERROR_CAUSE_CODE_DO_NOT_PRINT);
        else
            DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_CNET,ERR_CNET_INVALID_STREAM,0);

        bStream = 0;
    }

    //Bugzilla #2386
    if ((pkt[CNET_LEN_OFFSET] == 0x02)
        && (pkt[CNET_TYPE_OFFSET] == CNET_COMMAND)
        && (pkt[3] == CNET_CMD_ALL_CLEAR ))
    {
        //Bugzilla 30086 fix
        //Set error code to skip prints during CresnetConsole prints b/c infinite loop will occur
        if ( pkt[2] == RCON_PKT )
            DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_RESET_TX_QUEUE,DM_ERROR_CAUSE_CODE_DO_NOT_PRINT);
        else
            DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_RESET_TX_QUEUE,0);

        UartCresnetSlaveResetTxQueue(bStream);
        HwDelayMsec(300);
    }

    status = UartCresnetSlaveSendToTxQueue(bStream, (UINT8 *)packet);

    if (status != 0)
    {
        //Bugzilla 30086 fix
        //Set error code to skip prints during CresnetConsole prints b/c infinite loop will occur
        if ( pkt[2] == RCON_PKT )
            DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_RESET_TX_PIPE,DM_ERROR_CAUSE_CODE_DO_NOT_PRINT);
        else
            DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_RESET_TX_PIPE,0);

        //rwc: no longer restting tx queue here.  we only get here if we couldn't get a buffer to stick
        //  in the queue.  that's no reason to prevent other packets from going out.
    }

        return status;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveSetRestartFlag
 * \return      none
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveSetRestartFlag(BOOL set)
{
    m_restart = set;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       UartCresnetSlaveSetAutoId
 * \return      none
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveSetAutoId(UINT8 bStream)
{
    // get a pointer to this stream's parameters
    SLAVESTREAM *pStream = m_pStream;
    pStream += bStream;
    // set the autoid flag for it.
    pStream->autoid = 1;
}
/**
 * \author      Larry Salant
 * \date        12/19/2008
 * \brief       send the Indentity command on next poll
 * \return      stream number
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveSetIdentity(UINT8 bStream)
{
    // get a pointer to this stream's parameters
    SLAVESTREAM *pStream = m_pStream;
    pStream += bStream;
    // set the autoid flag for it.
    pStream->sendIdentity = 1;
}

void CDMUartCresnetSlave::UartCresnetSlaveGenerateBreak(void)
{

    UartCresnetSlaveSetNetmode(CNET_WAIT_BEFORE_BREAK);  // wait two character times before sending the break
    UartCresnetSlaveTimeNet(2*m_charTimeUsec);             // initialize timer to start bit time

}

/**
 * \author      Hazrat Shah
 * \date        3/15/2010
 * \brief       disable receiver interrupt
 * \return      void
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveDisableReceiver(void) {
        HwUartDisableReceiver(m_uart);
}


/**
 * \author      Hazrat Shah
 * \date        3/15/2010
 * \brief       enable receiver interrupt
 * \return      void
 * \retval      void
 * \param       void
 */
void CDMUartCresnetSlave::UartCresnetSlaveEnableReceiver(void)  {
        HwUartFlushReceiver(m_uart);
        HwUartEnableReceiver(m_uart);
}

/**
 * \author      Chris Merck
 * \date        20 June 2013
 * \brief       Reconfigure or disable serial port.
 * \param       BOOL bEnable: true = reconfigure and reenable, false = disable
 * \return      void
 * \retval      void
 */
void CDMUartCresnetSlave::Reconfigure(BOOL bEnable)
{
    OsEnterCritical();
    if (bEnable)
    {
      // zero buffers
        memset((void*)m_rxData, 0, sizeof(m_rxData));
        memset((void*)m_immediatePkt, 0, sizeof(m_immediatePkt));

		// restore comspec
        HwUartSetBaudRate(m_uart, m_bitsPerSec, 2);        
        
      // enable transmission
        m_bTxEnable = true;
        StartRespondingToPolls();

        //NOTE: reception enabled in ISR
    }
    else
    {
        // disable transmission        
        m_bTxEnable = false;
        StopRespondingToPolls();
    }
    OsExitCritical();
}
